import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/route/route.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_images.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/data/controller/reward/reward_controller.dart';
import 'package:viserpay/data/repo/reward/reward_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/bottom-sheet/custom_bottom_sheet.dart';
import 'package:viserpay/view/components/divider/custom_divider.dart';
import 'package:viserpay/view/screens/reward/widget/claim_bottom_sheet.dart';
import 'package:viserpay/view/screens/reward/widget/reward_card.dart';
import 'package:skeletonizer/skeletonizer.dart';

class RewardScreen extends StatefulWidget {
  const RewardScreen({super.key});

  @override
  State<RewardScreen> createState() => _RewardScreenState();
}

class _RewardScreenState extends State<RewardScreen> {
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(RewardRepo(apiClient: Get.find()));
    final controller = Get.put(RewardController(rewardRepo: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.initialize();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: MyColor.bgColor1,
      appBar: CustomAppBar(title: MyStrings.reward, isTitleCenter: true),
      body: GetBuilder<RewardController>(
        builder: (controller) {
          return RefreshIndicator(
            backgroundColor: MyColor.colorWhite,
            color: MyColor.primaryColor,
            onRefresh: () async {
              controller.initialize();
            },
            child: Skeletonizer(
              containersColor: MyColor.colorWhite.withValues(alpha: 0.05),
              effect: ShimmerEffect(baseColor: MyColor.colorGrey1, highlightColor: MyColor.colorGrey1),
              ignorePointers: true,
              enabled: controller.isLoading,
              child: SingleChildScrollView(
                padding: Dimensions.defaultPaddingHV,
                physics: const BouncingScrollPhysics(parent: AlwaysScrollableScrollPhysics()),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    InkWell(
                      onTap: () => Get.toNamed(RouteHelper.userRewardHistory),
                      child: Container(
                        width: double.infinity,
                        padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 20),
                        decoration: BoxDecoration(
                          borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                          gradient: LinearGradient(
                            colors: [MyColor.primaryColor, MyColor.primaryColor.withValues(alpha: .7)],
                            begin: Alignment.topLeft,
                            end: Alignment.bottomRight,
                          ),
                        ),
                        child: Column(
                          children: [
                            Row(
                              mainAxisAlignment: MainAxisAlignment.spaceBetween,
                              crossAxisAlignment: CrossAxisAlignment.center,
                              children: [
                                Text(controller.rewardRepo.apiClient.getCurrencyOrUsername(isCurrency: false, isSymbol: false).toCapitalized(), style: regularDefault.copyWith(color: MyColor.colorWhite)),
                                Row(
                                  children: [
                                    Image.asset(MyImages.star, width: Dimensions.space20, height: Dimensions.space20, color: MyColor.colorWhite),
                                    const SizedBox(width: Dimensions.space5),
                                    Text("${StringConverter.formatNumber(controller.user?.totalPoints ?? "0", precision: 0)} ${MyStrings.points.tr}", style: semiBoldDefault.copyWith(color: MyColor.colorWhite)),
                                  ],
                                ),
                              ],
                            ),
                            const SizedBox(height: Dimensions.space20),
                            Text(controller.getLevelByUserPoints().name ?? '', style: regularDefault.copyWith(color: MyColor.colorWhite), textAlign: TextAlign.center),
                            const SizedBox(height: Dimensions.space10),
                            LinearProgressIndicator(
                              value: controller.getProgressValue(controller.getLevelByUserPoints().maxPoints.toString()),
                              minHeight: 8,
                              backgroundColor: MyColor.colorWhite,
                              borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                              color: Colors.green,
                            ),
                            const SizedBox(height: Dimensions.space20),
                            Text(MyStrings.userPointSubtitle.replaceAll('_', StringConverter.formatNumber(controller.user?.totalPoints.toString() ?? "0.0", precision: 0)), style: lightDefault.copyWith(color: MyColor.colorWhite), textAlign: TextAlign.center),
                            const SizedBox(height: Dimensions.space10),
                          ],
                        ),
                      ),
                    ),
                    const SizedBox(height: Dimensions.space20),
                    Container(
                      width: double.infinity,
                      padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 20),
                      decoration: BoxDecoration(
                        borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                        color: MyColor.colorWhite,
                      ),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            children: [
                              Text("${MyStrings.appName.tr} ${MyStrings.reward.tr}", style: semiBoldDefault.copyWith(color: MyColor.primaryColor)),
                              const SizedBox(width: Dimensions.space10),
                              InkWell(
                                onTap: () => Get.toNamed(RouteHelper.rewardListScreen),
                                child: Text(MyStrings.viewAll, style: semiBoldDefault.copyWith(color: MyColor.primaryColor)),
                              ),
                            ],
                          ),
                          CustomDivider(),
                          Column(
                            children: List.generate(
                              controller.sortedRewards.length,
                              (index) {
                                bool isDisable = (double.tryParse(controller.user?.totalPoints ?? "0.0") ?? 0.0) < (double.tryParse(controller.sortedRewards[index].requiredPoints ?? "0.0") ?? 0.0);
                                return RewardCard(
                                  reward: controller.sortedRewards[index],
                                  isLoading: controller.sortedRewards[index].id == controller.selectedId,
                                  isUsed: controller.claimed(controller.sortedRewards[index].id.toString()),
                                  isDisable: isDisable,
                                  onTap: () {
                                    CustomBottomSheet(
                                      child: ClaimBottomSheet(
                                        reward: controller.sortedRewards[index],
                                        onTap: () {
                                          Get.back();
                                          controller.redeemReward(controller.sortedRewards[index]);
                                        },
                                      ),
                                    ).customBottomSheet(context);
                                  },
                                );
                              },
                            ),
                          ),
                        ],
                      ),
                    ),
                    const SizedBox(height: Dimensions.space20),
                    InkWell(
                      onTap: () => Get.toNamed(RouteHelper.userRewardHistory),
                      customBorder: RoundedRectangleBorder(borderRadius: BorderRadius.circular(Dimensions.largeRadius)),
                      child: Container(
                        width: double.infinity,
                        padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 15),
                        decoration: BoxDecoration(borderRadius: BorderRadius.circular(Dimensions.largeRadius), color: MyColor.colorWhite),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            Row(
                              children: [
                                Image.asset(MyImages.reward, width: Dimensions.space20, height: Dimensions.space20, color: MyColor.primaryColor),
                                SizedBox(width: Dimensions.space10),
                                Text(MyStrings.rewardHistory, style: regularDefault),
                              ],
                            ),
                            Icon(Icons.arrow_forward_ios, size: 20, color: MyColor.colorGrey.withValues(alpha: 0.5)),
                          ],
                        ),
                      ),
                    ),
                    const SizedBox(height: Dimensions.space20),
                    InkWell(
                      onTap: () => Get.toNamed(RouteHelper.rewardFaqScreen),
                      customBorder: RoundedRectangleBorder(borderRadius: BorderRadius.circular(Dimensions.largeRadius)),
                      child: Container(
                        width: double.infinity,
                        padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 15),
                        decoration: BoxDecoration(borderRadius: BorderRadius.circular(Dimensions.largeRadius), color: MyColor.colorWhite),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            Row(
                              children: [
                                Image.asset(MyImages.star, width: Dimensions.space20, height: Dimensions.space20, color: MyColor.primaryColor),
                                SizedBox(width: Dimensions.space10),
                                Text(MyStrings.rewardFaq, style: regularDefault),
                              ],
                            ),
                            Icon(Icons.arrow_forward_ios, size: 20, color: MyColor.colorGrey.withValues(alpha: 0.5)),
                          ],
                        ),
                      ),
                    ),
                  ],
                ),
              ),
            ),
          );
        },
      ),
    );
  }
}
